<?php
/* --------------------------------------------------------------
   ImageProcessingConfigurationRepository.php 2023-01-30
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2023 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\ImageProcessing\App\Data;

use Gambio\Admin\Modules\ImageProcessing\App\Creation\ImageProcessingConfigurationFactory;
use Gambio\Admin\Modules\ImageProcessing\Model\Configuration\CustomImageConfiguration;
use Gambio\Admin\Modules\ImageProcessing\Model\Configuration\GalleryImageConfiguration;
use Gambio\Admin\Modules\ImageProcessing\Model\Configuration\InfoImageConfiguration;
use Gambio\Admin\Modules\ImageProcessing\Model\Configuration\PopupImageConfiguration;
use Gambio\Admin\Modules\ImageProcessing\Model\Configuration\ThumbnailImageConfiguration;
use Gambio\Admin\Modules\ImageProcessing\Model\ImageProcessingSettings;
use Gambio\Core\Application\ValueObjects\Path;
use Gambio\Core\Configuration\Services\ConfigurationFinder;

/**
 * Class ImageProcessingConfigurationRepository
 *
 * @package Gambio\Admin\Modules\ImageProcessing\App\Data
 */
class ImageProcessingConfigurationRepository
{
    private const CONFIGURATION_PRODUCT_IMAGE_POPUP_HEIGHT     = 'configuration/PRODUCT_IMAGE_POPUP_HEIGHT';
    private const CONFIGURATION_PRODUCT_IMAGE_POPUP_WIDTH      = 'configuration/PRODUCT_IMAGE_POPUP_WIDTH';
    private const CONFIGURATION_PRODUCT_IMAGE_INFO_HEIGHT      = 'configuration/PRODUCT_IMAGE_INFO_HEIGHT';
    private const CONFIGURATION_PRODUCT_IMAGE_INFO_WIDTH       = 'configuration/PRODUCT_IMAGE_INFO_WIDTH';
    private const CONFIGURATION_PRODUCT_IMAGE_THUMBNAIL_HEIGHT = 'configuration/PRODUCT_IMAGE_THUMBNAIL_HEIGHT';
    private const CONFIGURATION_PRODUCT_IMAGE_THUMBNAIL_WIDTH  = 'configuration/PRODUCT_IMAGE_THUMBNAIL_WIDTH';
    private const PRODUCT_IMAGE_GALLERY_HEIGHT                 = 86;
    private const PRODUCT_IMAGE_GALLERY_WIDTH                  = 86;
    private const DIRECTORY_PRODUCT_IMAGES                     = '/images/product_images';
    
    private ImageProcessingConfigurationFactory $factory;
    private ConfigurationFinder                 $configurationFinder;
    private Path                                $path;
    
    
    /**
     * ImageProcessingConfigurationRepository constructor.
     *
     * @param ImageProcessingConfigurationFactory $factory
     * @param ConfigurationFinder                 $configurationFinder
     * @param Path                                $path
     */
    public function __construct(
        ImageProcessingConfigurationFactory $factory,
        ConfigurationFinder                 $configurationFinder,
        Path                                $path
    ) {
        $this->factory             = $factory;
        $this->configurationFinder = $configurationFinder;
        $this->path                = $path;
    }
    
    
    /**
     * Returns configuration for processing gallery images.
     *
     * @return GalleryImageConfiguration
     */
    public function getGalleryConfiguration(): GalleryImageConfiguration
    {
        $productImages = self::DIRECTORY_PRODUCT_IMAGES;
        $path          = "{$this->path->base()}$productImages/gallery_images";
        
        return $this->factory->createGalleryImageConfiguration(self::PRODUCT_IMAGE_GALLERY_HEIGHT,
                                                               self::PRODUCT_IMAGE_GALLERY_WIDTH,
                                                               $path);
    }
    
    
    /**
     * Returns configuration for processing info images.
     *
     * @return InfoImageConfiguration
     */
    public function getInfoImageConfiguration(): InfoImageConfiguration
    {
        $height = $this->configurationFinder->get(self::CONFIGURATION_PRODUCT_IMAGE_INFO_HEIGHT);
        $width  = $this->configurationFinder->get(self::CONFIGURATION_PRODUCT_IMAGE_INFO_WIDTH);
        
        $productImages = self::DIRECTORY_PRODUCT_IMAGES;
        $path          = "{$this->path->base()}$productImages/info_images";
        
        return $this->factory->createInfoImageConfiguration((int)$height, (int)$width, $path);
    }
    
    
    /**
     * Returns configuration for processing popup images.
     *
     * @return PopupImageConfiguration
     */
    public function getPopupConfiguration(): PopupImageConfiguration
    {
        $height = $this->configurationFinder->get(self::CONFIGURATION_PRODUCT_IMAGE_POPUP_HEIGHT);
        $width  = $this->configurationFinder->get(self::CONFIGURATION_PRODUCT_IMAGE_POPUP_WIDTH);
        
        $productImages = self::DIRECTORY_PRODUCT_IMAGES;
        $path          = "{$this->path->base()}$productImages/popup_images";
        
        return $this->factory->createPopupImageConfiguration((int)$height, (int)$width, $path);
    }
    
    
    /**
     * Returns configuration for processing thumbnail images.
     *
     * @return ThumbnailImageConfiguration
     */
    public function getThumbnailConfiguration(): ThumbnailImageConfiguration
    {
        $height = $this->configurationFinder->get(self::CONFIGURATION_PRODUCT_IMAGE_THUMBNAIL_HEIGHT);
        $width  = $this->configurationFinder->get(self::CONFIGURATION_PRODUCT_IMAGE_THUMBNAIL_WIDTH);
        
        $productImages = self::DIRECTORY_PRODUCT_IMAGES;
        $path          = "{$this->path->base()}$productImages/thumbnail_images";
        
        return $this->factory->createThumbnailImageConfiguration((int)$height, (int)$width, $path);
    }
    
    
    /**
     * Returns configuration for processing custom image files.
     *
     * @param ImageProcessingSettings $settings
     *
     * @return CustomImageConfiguration
     */
    public function getCustomConfiguration(ImageProcessingSettings $settings): CustomImageConfiguration
    {
        return $this->factory->createCustomImageConfiguration($settings->newHeight(),
                                                              $settings->newWidth(),
                                                              $settings->sourcePath(),
                                                              $settings->destinationPath());
    }
}